/*****************************************************************************
*                           Freescale Semiconductor                          *
*                                                                            *
*  Project    : AN3815 - MC56F8006 Modular Pixel Matrix                      *
*  Version    : 0.1                                                          *
*  Date       : 16/Apr/2009                                                  *
*  Authors    : Alexandre Dias, Bruno Bastos                                 *
*               Humberto Carvalho, Leonardo Mangiapelo                       *
*               Renato Frias                                                 *
*                                                                            *
* -------------------------------------------------------------------------- *
* THIS SOFTWARE IS PROVIDED BY FREESCALE "AS IS" AND ANY EXPRESSED OR        *
* IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES  *
* OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.    *
* IN NO EVENT SHALL FREESCALE OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT,  *
* INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES         *
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR         *
* SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)         *
* HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,        * 
* STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING      *
* IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF             *
* THE POSSIBILITY OF SUCH DAMAGE.                                            *
* -------------------------------------------------------------------------- *
*  Description:                                                              *
* This software complements AN3815 - Implementing a Modular High Brightness  *
* RGB LED Network. The aim of the code is to proof the concept described on  *
* the application note document. It also serves as a starting point for      * 
* applications with MC56F8006 and some of its peripherals                    *
* -------------------------------------------------------------------------- *
*                                                                            *
*  File: adc.c	                                                             *
*                                                                            *
*    Interfaces with ADC registers on 56F8006.								 * 
*                                                                            *
******************************************************************************/

/************************* Include Files *************************************/

#include "util.h"
#include "adc.h"
#include "pwm.h"

/******************************** Variables **********************************/

//! ADC Data buffer for the green color.
uint16_t adc_green_buffer[ADC_BUFFER_SIZE];
//! ADC Data buffer for the red color.
uint16_t adc_red_buffer[ADC_BUFFER_SIZE];
//! Data buffer for the blue color.
uint16_t adc_blue_buffer[ADC_BUFFER_SIZE];
//! ADC task state machine controller.
adc_state_machine state_machine;
//! poiter for ADC data buffers.
uint8_t adc_count;

/*********************** Function's Implementation ***************************/

//! Initializes ADC.
/*! Set all ADC related functionality.*/
void adc_init()
{

 state_machine = READ_GREEN;
 adc_count = 0;
	
 // Clock Gate
 SIM_PCE |= SIM_PCE_ADC0;
 
 // GPIO PER
 GPIOC_PER |= GPIOC_PER_PER_0 | GPIOC_PER_PER_1 | GPIOC_PER_PER_2;
  
 // Pin Muxing
 SIM_GPSC &= ~SIM_GPSC_GPS_C0;
 
 //Initialize the ADC
 // ADLPC   = 0  (high speed)
 // ADIV    = 11 (div by 1)
 // ADLSMP  = 1  (short sample)
 // MODE    = 01 (12-bit mode)
 // ADICLK  = 00 (busclk)
  
 ADC0_ADCCFG =  ADC0_ADCCFG_ADLSMP | ADC0_ADCCFG_ADIV | ADC0_ADCCFG_MODE_0;
 ADC0_ADCSC1A = ADC0_ADCSC1A_AIEN | ADC_GREEN_CHANNEL;
}

//! Returns an ADC reads average for a given color
/*! 
	\param color the color for which the average should be calculated.
	\return color average that depends on buffer size. 
	
	Sums all ADC samples taken and returns their average.
	
*/
uint16_t adc_get_color_value(adc_colors color)
{
	uint8_t i;
	uint16_t result;
	
	result = 0;
	switch(color)
	{
		case ADC_RED:
		for (i=0;i<ADC_BUFFER_SIZE;i++)
		{
			result += adc_red_buffer[i];
		}
		break;
		
		case ADC_GREEN:
		for (i=0;i<ADC_BUFFER_SIZE;i++)
		{
			result += adc_green_buffer[i];
		}
		break;
		
		case ADC_BLUE:
		for (i=0;i<ADC_BUFFER_SIZE;i++)
		{
			result += adc_blue_buffer[i];
		}
		break;
		
	
	}
		result = result / ADC_BUFFER_SIZE;
		return result;
}


//! ADC convertion complete interrupt.
/*! 
	Stores ADC samples and color buffers.	
*/

#pragma interrupt on
#pragma section INTERRUPT_SERVICE_ROUTINES begin

void adc_isr(void)
{
	switch(state_machine)
		{
			case READ_GREEN:
		
			adc_green_buffer[adc_count] = ADC0_ADCRA >> 3;
			state_machine = READ_RED;
			ADC0_ADCSC1A = ADC_RED_CHANNEL;
			break;
		
			case READ_RED:
			
			adc_red_buffer[adc_count] = ADC0_ADCRA >> 3;
  			state_machine = READ_BLUE;
  			ADC0_ADCSC1A = ADC_BLUE_CHANNEL;
  			  			
			break;
		
			case READ_BLUE:
			adc_blue_buffer[adc_count] = ADC0_ADCRA >> 3;
  			adc_count++;
  			if(adc_count == ADC_BUFFER_SIZE)
  			{
  				adc_count = 0;
  			}
  			ADC0_ADCSC1A = ADC_GREEN_CHANNEL;
  			state_machine = READ_GREEN;
			break;
		}
		ADC0_ADCSC1A |= ADC0_ADCSC1A_AIEN;	
}
#pragma section INTERRUPT_SERVICE_ROUTINES end
